import { randomUUID } from "crypto";
import { Role } from "@prisma/client";
import { NextResponse } from "next/server";
import { prisma } from "@/lib/prisma";
import { requireRole } from "@/lib/rbac";

export const dynamic = "force-dynamic";

export async function POST(_: Request, { params }: { params: { id: string } }) {
  const guard = await requireRole([Role.ADMIN]);
  if (guard.error || !guard.session) return guard.error;

  const token = randomUUID();
  const expiresAt = new Date(Date.now() + 1000 * 60 * 30);

  await prisma.passwordResetToken.create({
    data: {
      token,
      userId: params.id,
      expiresAt
    }
  });

  await prisma.auditLog.create({
    data: {
      actorId: guard.session.sub,
      action: "PASSWORD_RESET_LINK_TRIGGERED",
      target: params.id,
      description: "Password reset link generated by admin"
    }
  });

  const resetLink = `${process.env.APP_URL ?? "http://localhost:3000"}/auth/reset-password?token=${token}`;
  return NextResponse.json({ ok: true, resetLink });
}
